<?php

namespace idoit\Module\SyneticsJdisc\Controller;

use idoit\Module\SyneticsJdisc\Controller\Response\ArrayResponse;
use idoit\Module\SyneticsJdisc\Controller\Response\NotFoundResponse;
use idoit\Module\SyneticsJdisc\Controller\Table\DeviceListSearchParams;
use idoit\Module\SyneticsJdisc\Model\DeviceListDao;
use isys_application;
use isys_cmdb_dao;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

class DeviceListController extends AbstractController
{
    private ?DeviceListDao $dao = null;

    /**
     * @return DeviceListDao
     * @throws \Exception
     */
    private function getDao(): DeviceListDao
    {
        if (null === $this->dao) {
            $this->dao = isys_application::instance()->container->get('idoit.jdisc.device-list.dao');
        }
        return $this->dao;
    }

    /**
     * @param Request $request
     * @return Response
     */
    public function find(Request $request): Response
    {
        $serverId = $request->get('serverId');
        $search   = DeviceListSearchParams::fromParams($request->get('params'));

        $dao = $this->getDao();
        if (!$dao->init($serverId)) {
            return new NotFoundResponse('DiscoveryServer');
        }

        $devices = $dao->getDevices($search);

        return new ArrayResponse(
            $devices['rows'],
            $devices['total']
        );
    }

    /**
     * @param int $id
     * @param Request $request
     * @return Response
     */
    public function get(int $id, Request $request): Response
    {
        $serverId = $request->get('serverId');

        $dao = $this->getDao();
        if (!$dao->init($serverId)) {
            return new NotFoundResponse('DiscoveryServer');
        }

        $device = $this->getDao()->get($id);
        if (empty($device)) {
            return new NotFoundResponse('Device');
        }
        return new JsonResponse($device);
    }

    /**
     * @param int $id
     * @param Request $request
     * @return Response
     */
    public function delete(int $id, Request $request): Response
    {
        $serverId = $request->get('serverId');

        $dao = $this->getDao();
        if (!$dao->init($serverId)) {
            return new NotFoundResponse('DiscoveryServer');
        }

        $result = $this->getDao()->delete($id);
        if (!$result) {
            return new NotFoundResponse('Device');
        }
        return new JsonResponse($result);
    }

    /**
     * @return JsonResponse
     */
    public function getIdoitTypes(): JsonResponse
    {
        $result = isys_cmdb_dao::instance(isys_application::instance()->container->get('database'))
            ->get_objtype(null, false, C__RECORD_STATUS__NORMAL);
        $language = isys_application::instance()->container->get('language');
        $objectTypes = [];

        while ($row = $result->get_row()) {
            $objectTypes[$row['isys_obj_type__id']] = $language->get($row['isys_obj_type__title']);
        }

        asort($objectTypes);

        return new JsonResponse($objectTypes);
    }
}
