<?php

namespace idoit\Module\SyneticsJdisc\Controller;

use idoit\Module\SyneticsJdisc\Controller\Response\ArrayResponse;
use idoit\Module\SyneticsJdisc\Controller\Response\NotFoundResponse;
use idoit\Module\SyneticsJdisc\Controller\Table\SearchParams;
use idoit\Module\SyneticsJdisc\Model\JDiscLogDao;
use idoit\Module\SyneticsJdisc\View\Logs;
use isys_application;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

class LogsController extends AbstractController
{
    private ?JDiscLogDao $dao = null;

    /**
     * @return JDiscLogDao
     * @throws \Exception
     */
    private function getDao(): JDiscLogDao
    {
        if (null === $this->dao) {
            $this->dao = isys_application::instance()->container->get('idoit.jdisc.log.dao');
        }
        return $this->dao;
    }

    /**
     * @param Request $request
     * @return Response
     */
    public function find(Request $request): Response
    {
        $search = SearchParams::fromParams($request->get('params'));

        return new ArrayResponse(
            $this->getDao()->getData(null, $search),
            $this->getDao()->getCount(),
        );
    }

    /**
     * @param int $id
     * @return Response
     */
    public function get(int $id): Response
    {
        $log = $this->getDao()->get($id);
        if ($log === null) {
            return new NotFoundResponse('Log');
        }
        return new JsonResponse($log);
    }

    /**
     * @param int $id
     * @return Response
     */
    public function delete(int $id): Response
    {
        $result = $this->getDao()->delete($id);
        if (!$result) {
            return new NotFoundResponse('Log');
        }
        return new JsonResponse(['success' => true]);
    }

    /**
     * @param Request $request
     * @param int|null $id
     * @return Response
     */
    public function save(Request $request, ?int $id): Response
    {
        $data = json_decode($request->getContent(), true);
        if (empty($data)) {
            return new JsonResponse(['error' => 'No data provided'], Response::HTTP_BAD_REQUEST);
        }

        if (!empty($id)) {
            $data['id'] = $id;
        } else {
            unset($data['id']);
        }

        $logId = $this->getDao()->save($data);
        if (empty($logId)) {
            return new JsonResponse(['error' => 'Log save error'], Response::HTTP_INTERNAL_SERVER_ERROR);
        }

        return new JsonResponse(['id' => $logId]);
    }

    /**
     * @param int $id
     * @return Response
     */
    public function download(int $id): Response
    {
        $url = isys_application::instance()->container->get('route_generator')->generate(
            'system.file-download',
            [
                'type'       => 'jdisc.log-file',
                'identifier' => $id,
            ]
        );
        if (!$url) {
            return new JsonResponse(['error' => 'Log download error'], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
        return new JsonResponse(['url' => $url]);
    }

    /**
     * @return Response
     * @throws \Exception
     */
    public function page(): Response
    {
        return Logs::factory()->render();
    }
}
