<?php

namespace idoit\Module\SyneticsJdisc\Controller;

use idoit\Module\SyneticsJdisc\Controller\Response\NotFoundResponse;
use idoit\Module\SyneticsJdisc\Model\DeviceListDao;
use isys_application;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

class ImportController extends AbstractController
{
    private ?DeviceListDao $dao = null;

    private const IMPORT_MODES = [
        'create',
        'create_only_new_devices',
        'update',
        'update_new_discovery',
        'update_existing_only',
        'overwrite',
        'overwrite_new_discovery',
    ];

    /**
     * @return DeviceListDao
     * @throws \Exception
     */
    private function getDao(): DeviceListDao
    {
        if (null === $this->dao) {
            $this->dao = isys_application::instance()->container->get('idoit.jdisc.device-list.dao');
        }
        return $this->dao;
    }

    /**
     * @param Request $request
     * @return Response
     */
    public function statistics(Request $request): Response
    {
        $serverId   = $request->get('serverId');
        $profileId  = $request->get('sync_profile');
        $importMode = $request->get('import_mode');
        $group      = $request->get('group');

        if (empty($profileId) || empty($importMode) || !in_array($importMode, self::IMPORT_MODES)) {
            return new JsonResponse(['error' => 'No data provided'], Response::HTTP_BAD_REQUEST);
        }

        $dao = $this->getDao();
        if (!$dao->init($serverId)) {
            return new NotFoundResponse('DiscoveryServer');
        }

        $statistics = $dao->getStatistics($profileId, $importMode, $group);

        return new JsonResponse([
            'results' => $statistics,
            'total'   => count($statistics),
        ]);
    }

    /**
     * @param Request $request
     * @return Response
     */
    public function settings(Request $request): Response
    {
        $dao = $this->getDao();
        if (!$dao->init(null)) {
            $errorResponse = new NotFoundResponse('DiscoveryServer');
            $errorResponse->setStatusCode(Response::HTTP_OK);
            return $errorResponse;
        }
        $defaultServer = $dao->getServerId();

        $language = isys_application::instance()->container->get('language');
        $tenantSettings = isys_application::instance()->container->get('settingsTenant');

        $maxImportPause = (int) $tenantSettings->get('jdisc.max_pause_time_before_action', 15);
        $maxImportPause = $maxImportPause > 1 ? $maxImportPause : 1;
        $afterPauseAction = $tenantSettings->get('jdisc.after_pause_action', 'cancel') === 'cancel'
            ? $language->get('LC__UNIVERSAL__BUTTON_CANCEL')
            : $language->get('LC__UNIVERSAL__BUTTON_CONTINUE');

        return new JsonResponse([
            'defaultServer'    => $defaultServer,
            'maxImportPause'   => $maxImportPause,
            'afterPauseAction' => $afterPauseAction,
        ]);
    }
}
